package org.apache.shardingsphere.transaction.base.seata.at;

import com.google.common.base.Preconditions;
import io.seata.config.FileConfiguration;
import io.seata.core.context.RootContext;
import io.seata.rm.RMClient;
import io.seata.rm.datasource.DataSourceProxy;
import io.seata.tm.TMClient;
import io.seata.tm.api.GlobalTransaction;
import io.seata.tm.api.GlobalTransactionContext;
import lombok.SneakyThrows;
import org.apache.shardingsphere.spi.database.type.DatabaseType;
import org.apache.shardingsphere.transaction.core.ResourceDataSource;
import org.apache.shardingsphere.transaction.core.TransactionType;
import org.apache.shardingsphere.transaction.spi.ShardingTransactionManager;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * Seata AT sharding transaction manager.
 */
public final class SeataATShardingTransactionManager implements ShardingTransactionManager {

    private final Map<String, DataSource> dataSourceMap = new HashMap<>();

    private final String applicationId;

    private final String transactionServiceGroup;

    private final boolean enableSeataAT;

    public SeataATShardingTransactionManager() {
        FileConfiguration configuration = new FileConfiguration("seata.conf");
        enableSeataAT = configuration.getBoolean("client.sharding.transaction.seata.at.enable", false);
        applicationId = configuration.getConfig("client.application.id");
        transactionServiceGroup = configuration.getConfig("client.transaction.service.group", "default");
    }

    @Override
    public void init(final DatabaseType databaseType, final Collection<ResourceDataSource> resourceDataSources) {
        if (enableSeataAT) {
            initSeataRPCClient();
            for (ResourceDataSource each : resourceDataSources) {
                dataSourceMap.put(each.getOriginalName(), new DataSourceProxy(each.getDataSource()));
            }
        }
    }

    private void initSeataRPCClient() {
        Preconditions.checkNotNull(applicationId, "please config application id within seata.conf file.");
        TMClient.init(applicationId, transactionServiceGroup);
        RMClient.init(applicationId, transactionServiceGroup);
    }

    @Override
    public TransactionType getTransactionType() {
        return TransactionType.BASE;
    }

    @Override
    public boolean isInTransaction() {
        Preconditions.checkState(enableSeataAT, "sharding seata-at transaction has been disabled.");
        return null != RootContext.getXID();
    }

    @Override
    public Connection getConnection(final String dataSourceName) throws SQLException {
        Preconditions.checkState(enableSeataAT, "sharding seata-at transaction has been disabled.");
        return dataSourceMap.get(dataSourceName).getConnection();
    }

    @Override
    @SneakyThrows
    public void begin() {
        Preconditions.checkState(enableSeataAT, "sharding seata-at transaction has been disabled.");
        GlobalTransaction globalTransaction = GlobalTransactionContext.getCurrentOrCreate();
        globalTransaction.begin();
        SeataTransactionHolder.set(globalTransaction);
    }

    @Override
    @SneakyThrows
    public void commit() {
        Preconditions.checkState(enableSeataAT, "sharding seata-at transaction has been disabled.");
        try {
            if(SeataTransactionHolder.get()==null){
                GlobalTransaction globalTransaction = GlobalTransactionContext.getCurrentOrCreate();
                //全局事务已经开启不需要启动
                if(globalTransaction.getStatus()!=null&&globalTransaction.getStatus().getCode()==1){
                    SeataTransactionHolder.set(globalTransaction);
                }else{
                    globalTransaction.begin();
                    SeataTransactionHolder.set(globalTransaction);
                }
            }
            SeataTransactionHolder.get().commit();
        } finally {
            SeataTransactionHolder.clear();
            RootContext.unbind();
        }
    }

    @Override
    @SneakyThrows
    public void rollback() {
        Preconditions.checkState(enableSeataAT, "sharding seata-at transaction has been disabled.");
        try {
            if(SeataTransactionHolder.get()==null){
                GlobalTransaction globalTransaction = GlobalTransactionContext.getCurrentOrCreate();
                //全局事务已经开启不需要启动
                if(globalTransaction.getStatus()!=null&&globalTransaction.getStatus().getCode()==1){
                    SeataTransactionHolder.set(globalTransaction);
                }else{
                    globalTransaction.begin();
                    SeataTransactionHolder.set(globalTransaction);
                }
            }
            SeataTransactionHolder.get().rollback();
        } finally {
            SeataTransactionHolder.clear();
            RootContext.unbind();
        }
    }

    @Override
    public void close() {
        dataSourceMap.clear();
        SeataTransactionHolder.clear();
    }
}
