package com.elitesland.sale.api.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.sale.api.vo.param.taskinfo.TaskInfoQueryVO;
import com.elitesland.sale.api.vo.resp.taskinfo.TaskInfoRespVO;
import com.elitesland.sale.api.vo.save.TaskInfoSaveVO;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 功能说明:业务员任务管理
 * </p>
 *
 * @Author Darren
 * @Date 2023/04/10
 * @Version 1.0
 * @Content:
 */
public interface TaskInfoService {

    /**
     * 业务员任务新增时保存/发布
     *
     * @param saveVO 入参
     * @return 任务出参对象
     */
    TaskInfoRespVO createTask(TaskInfoSaveVO saveVO);

    /**
     * 业务员任务修改时保存/发布
     *
     * @param saveVO 入参
     * @return 任务出参对象
     */
    TaskInfoRespVO updateTask(TaskInfoSaveVO saveVO);

    /**
     * 任务分页查询
     *
     * @param pageParam 入参
     * @return 任务信息集合
     */
    PagingVO<TaskInfoRespVO> page(TaskInfoQueryVO pageParam);

    /**
     * 根据任务ID查询任务详情数据
     *
     * @param id 任务ID
     * @return 任务详情数据
     */
    TaskInfoRespVO findIdOne(Long id);

    /**
     * 根据任务ID查询任务主表数据
     *
     * @param id 任务ID
     * @return 任务主表数据
     */
    TaskInfoRespVO findById(Long id);

    /**
     * 根据任务编码查询任务详情数据
     *
     * @param code 任务编码
     * @return 任务详情数据
     */
    TaskInfoRespVO findCodeOne(String code);

    /**
     * 根据任务编码查询任务主表数据
     *
     * @param code 根据任务编码
     * @return 任务主表数据
     */
    TaskInfoRespVO findByCode(String code);

    /**
     * 根据入参查询任务主表数据
     *
     * @param queryVO 入参
     * @return 任务主表数据
     */
    List<TaskInfoRespVO> selectByParam(TaskInfoQueryVO queryVO);

    /**
     * 根据入参查询任务主表数据-带分页
     *
     * @param queryVO 入参
     * @return 任务主表数据
     */
    List<TaskInfoRespVO> selectPageByQueryVO(TaskInfoQueryVO queryVO);

    /**
     * 根据任务ID批量删除
     *
     * @param ids 任务ID集合
     * @return
     */
    void deleteBatch(List<Long> ids);

    /**
     * 根据任务ID取消
     *
     * @param id 任务ID
     * @return
     */
    void cancelTask(Long id);

    /**
     * 根据任务ID发布
     *
     * @param id 任务ID
     * @return
     */
    void releaseTask(Long id);

    /**
     * 根据任务ID完成前置校验
     *
     * @param id 任务ID
     * @return true 校验通过  false 校验未通过
     */
    Boolean accomplishCheckTask(Long id);

    /**
     * 根据执行记录编码查询执行记录信息
     *
     * @param recordCode 执行记录编码
     * @return 有关联的执行记录为true, 否则为false
     */
    boolean selectExecutionRecordCheck(List<String> recordCode);

    /**
     * 根据任务ID完成
     *
     * @param id 任务ID
     * @return
     */
    void accomplishTask(Long id);

    /**
     * 根据任务ID关闭前置校验
     *
     * @param id 任务ID
     * @return true 校验通过  false 校验未通过
     */
    Boolean closeCheckTask(Long id);

    /**
     * 根据任务ID关闭
     *
     * @param id 任务ID
     * @return
     */
    void closeTask(Long id);

    /**
     * 判断任务是否逾期,并更新是否逾期
     *
     * @param queryVO 入参
     * @return 已逾期的任务ID
     */
    List<Long> overdueTask(TaskInfoQueryVO queryVO);

    /**
     * 根据任务ID更新任务状态
     *
     * @param state 任务状态
     * @param id    任务ID
     * @return
     */
    void updateStateById(String state, Long id);

    /**
     * 根据任务ID更新任务状态、完成时间
     *
     * @param state        任务状态
     * @param completeTime 完成时间
     * @param id           任务ID
     * @return
     */
    void updateStateAndCompleteTime(String state, LocalDateTime completeTime, Long id);


    /**
     * 根据任务ID更新任务进度
     *
     * @param progress 任务进度
     * @param id       任务ID
     * @return
     */
    void updateProgressById(BigDecimal progress, Long id);


}
