package com.elitesland.zhiyuan.openapi.outer.jushuitan.service.impl;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.sys.outlog.param.PushRecordDTO;
import com.elitescloud.cloudt.sys.outlog.rpc.OutsidePushRecordRepository;
import com.elitesland.zhiyuan.openapi.outer.jggyun.params.SupplierRequest;
import com.elitesland.zhiyuan.openapi.outer.jggyun.params.UpdateGoodsRequest;
import com.elitesland.zhiyuan.openapi.outer.jggyun.result.UpdateGoodsResponse;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.service.JushuitaService;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.util.NotaoUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
@RefreshScope
public class JushuitaServiceImpl implements JushuitaService {

    private final OutsidePushRecordRepository outsidePushRecordRepository;

    @Override
    @Transactional
    public String createMaterial(UpdateGoodsRequest request) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getItems())) {
            String errorMsg = "createMaterial请求参数为空或items列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        if (request.getItems().size() > 1) {
            log.warn("createMaterial请求包含{}个商品，仅处理第一个商品。建议每次请求仅包含一个商品", request.getItems().size());
        }

        // 2. 关键日志：记录原始请求
        log.info("聚水潭物料接口 - 请求开始 | 商品ID: {}, 请求参数: {}",
                request.getItems().get(0).getSkuId(),
                JSON.toJSONString(request, true)); // 详细打印避免信息丢失

        PushRecordDTO pushRecordDTO = buildPushRecordDTO("JST", "createMaterial");
        pushRecordDTO.setReqBody(request);
        try {
            // 3. 构建请求数据
            Map<String, Object> itemMap = buildItemMap(request);
            Map<String, Object> requestMap = Collections.singletonMap("items", Collections.singletonList(itemMap));
            String jsonRequest = JSON.toJSONString(requestMap);

            log.debug("聚水潭物料接口 - 请求数据: {}", jsonRequest);
            pushRecordDTO.setReqBody(requestMap); // 保存结构化请求

            // 4. 调用外部接口
            String response = NotaoUtil.getSellerItem(jsonRequest, "/open/jushuitan/itemsku/upload");
            log.debug("聚水潭物料接口 - 响应: {}", response);

            // 5. 处理响应
            if (response == null) {
                log.error("聚水潭物料接口 - 响应为空");
                pushRecordDTO.setReqSuccess(false);
                pushRecordDTO.setErrorMsg("接口返回为空");
                throw new RuntimeException("聚水潭接口返回为空");
            }

            UpdateGoodsResponse responseObj = JSON.parseObject(response, UpdateGoodsResponse.class);
            boolean success = handleResponseData(responseObj, pushRecordDTO);

            pushRecordDTO.setReqSuccess(success);
            pushRecordDTO.setRespBody(response);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);

            log.info("聚水潭物料接口 - 处理完成 | 商品ID: {}, 响应状态: {}",
                    request.getItems().get(0).getSkuId(),
                    success ? "成功" : "失败");
            return response;

        } catch (Exception e) {
            log.error("聚水潭物料接口调用异常 | 商品ID: {}", request.getItems().get(0).getSkuId(), e);
            pushRecordDTO.setErrorMsg(e.getMessage());
            pushRecordDTO.setReqSuccess(false);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            throw new Exception("聚水潭物料接口调用失败: " + e.getMessage(), e);
        }
    }

    private boolean handleResponseData(UpdateGoodsResponse responseObj, PushRecordDTO pushRecordDTO) {
        if (responseObj == null || responseObj.getData() == null || CollectionUtils.isEmpty(responseObj.getData().getDatas())) {
            log.warn("聚水潭物料接口响应数据为空");
            return false;
        }

        boolean success = true;
        List<String> failedItems = new ArrayList<>();
        for (UpdateGoodsResponse.Datas data : responseObj.getData().getDatas()) {
            if (!data.isSuccess()) {
                success = false;
                failedItems.add(data.getSkuId());
            }
        }

        if (!success) {
            log.warn("聚水潭物料接口部分商品失败 | 失败商品ID: {}",
                    failedItems.stream().collect(Collectors.joining(",")));
        }
        return success;
    }

    private Map<String, Object> buildItemMap(UpdateGoodsRequest request) {
        UpdateGoodsRequest.Item item = request.getItems().get(0);
        Map<String, Object> itemMap = new HashMap<>();

        // 必填字段验证
        if (StringUtils.isEmpty(item.getSkuId()) || StringUtils.isEmpty(item.getIId()) || StringUtils.isEmpty(item.getName())) {
            throw new IllegalArgumentException(String.format(
                    "物料必填字段缺失 | skuId: %s, iId: %s, name: %s",
                    item.getSkuId(), item.getIId(), item.getName()
            ));
        }

        itemMap.put("sku_id", item.getSkuId());
        itemMap.put("i_id", item.getIId());
        itemMap.put("name", item.getName());
        return itemMap;
    }

    @Override
    @Transactional
    public String syncSupp(SupplierRequest request) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getSuppliers())) {
            String errorMsg = "syncSupp请求参数为空或suppliers列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        log.info("聚水潭供应商同步接口 - 请求开始 | 供应商数量: {}", request.getSuppliers().size());
        PushRecordDTO pushRecordDTO = buildPushRecordDTO("JST", "syncSupp");
        pushRecordDTO.setReqBody(request);

        try {
            // 2. 构建供应商列表
            List<Map<String, Object>> suppliers = request.getSuppliers().stream()
                    .map(this::buildSupplierItemMap)
                    .collect(Collectors.toList());

            String jsonRequest = JSON.toJSONString(suppliers);
            log.debug("聚水潭供应商同步接口 - 请求数据: {}", jsonRequest);
            pushRecordDTO.setReqBody(suppliers); // 保存结构化请求

            // 3. 调用外部接口
            String response = NotaoUtil.getSellerItem(jsonRequest, "/open/supplier/upload");
            log.debug("聚水潭供应商同步接口 - 响应: {}", response);

            // 4. 处理响应
            if (response == null) {
                log.error("聚水潭供应商同步接口 - 响应为空");
                pushRecordDTO.setReqSuccess(false);
                pushRecordDTO.setErrorMsg("接口返回为空");
                throw new RuntimeException("聚水潭接口返回为空");
            }

            UpdateGoodsResponse responseObj = JSON.parseObject(response, UpdateGoodsResponse.class);
            boolean success = handleResponseData(responseObj, pushRecordDTO);

            pushRecordDTO.setReqSuccess(success);
            pushRecordDTO.setRespBody(response);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);

            log.info("聚水潭供应商同步接口 - 处理完成 | 供应商数量: {}, 状态: {}",
                    request.getSuppliers().size(), success ? "成功" : "失败");
            return response;

        } catch (Exception e) {
            log.error("聚水潭供应商同步接口异常 | 供应商数量: {}", request.getSuppliers().size(), e);
            pushRecordDTO.setErrorMsg(e.getMessage());
            pushRecordDTO.setReqSuccess(false);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            throw new Exception("聚水潭供应商同步接口失败: " + e.getMessage(), e);
        }
    }

    private PushRecordDTO buildPushRecordDTO(String interType, String method) {
        PushRecordDTO dto = new PushRecordDTO();
        dto.setInterType(interType);
        dto.setDocNo("");
        dto.setDocId(0L);
        dto.setReqSuccess(false);
        return dto;
    }

    private Map<String, Object> buildSupplierItemMap(SupplierRequest.SupplierItem item) {
        Map<String, Object> map = new HashMap<>();
        map.put("supplier_code", item.getSupplier_code());
        map.put("name", item.getName());
        map.put("enabled", true);
        return map;
    }
}