package com.elitesland.zhiyuan.openapi.outer.jushuitan.service.impl;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.sys.outlog.OutsidePushInter;
import com.elitescloud.cloudt.sys.outlog.param.PushRecordDTO;
import com.elitescloud.cloudt.sys.outlog.rpc.OutsidePushRecordRepository;
import com.elitesland.zhiyuan.openapi.outer.jggyun.params.SupplierRequest;
import com.elitesland.zhiyuan.openapi.outer.jggyun.params.UpdateGoodsRequest;
import com.elitesland.zhiyuan.openapi.outer.jggyun.result.UpdateGoodsResponse;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.common.service.JstService;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.params.req.PurchaseInboundListRequest;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.params.resp.JushuitanBaseResponse;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.params.resp.PurchaseInboundResponse;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.service.JushuitaService;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.util.NotaoUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
@RefreshScope
public class JushuitaServiceImpl implements JushuitaService {

    private final OutsidePushRecordRepository outsidePushRecordRepository;
    private final JstService jstService;

    @Override
    @Transactional
    public JushuitanBaseResponse createMaterial(UpdateGoodsRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getItems())) {
            String errorMsg = "createMaterial请求参数为空或items列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        if (request.getItems().size() > 1) {
            log.warn("createMaterial请求包含{}个商品，仅处理第一个商品。建议每次请求仅包含一个商品", request.getItems().size());
        }

        // 2. 关键日志：记录原始请求
        log.info("聚水潭物料接口 - 请求开始 | 商品ID: {}, 请求参数: {}",
                request.getItems().get(0).getSkuId(),
                JSON.toJSONString(request, true)); // 详细打印避免信息丢失

        PushRecordDTO pushRecordDTO = buildPushRecordDTO(pushRecordInterType, docNo, docId);
        pushRecordDTO.setReqBody(request);
        try {
            // 3. 构建请求数据
            Map<String, Object> itemMap = buildItemMap(request);
            Map<String, Object> requestMap = Collections.singletonMap("items", Collections.singletonList(itemMap));
            String jsonRequest = JSON.toJSONString(requestMap);

            log.debug("聚水潭物料接口 - 请求数据: {}", jsonRequest);
            pushRecordDTO.setReqBody(requestMap); // 保存结构化请求

            JushuitanBaseResponse response = jstService.getSellerItem(jsonRequest, "/open/jushuitan/itemsku/upload", OutsidePushInter.INV_IDO_SYNC_TO_JST.name(), 11L, "22");

            log.debug("聚水潭物料接口 - 响应: {}", response);

            // 5. 处理响应
            if (response == null) {
                log.error("聚水潭物料接口 - 响应为空");
                pushRecordDTO.setReqSuccess(false);
                pushRecordDTO.setErrorMsg("接口返回为空");
                throw new RuntimeException("聚水潭接口返回为空");
            }
            pushRecordDTO.setReqSuccess(true);
            pushRecordDTO.setRespBody(response);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            log.info("聚水潭物料接口 - 处理完成 | 商品ID: {}, 响应状态: {}",
                    request.getItems().get(0).getSkuId(),
                    "成功");
            return response;

        } catch (Exception e) {
            log.error("聚水潭物料接口调用异常 | 商品ID: {}", request.getItems().get(0).getSkuId(), e);
            pushRecordDTO.setErrorMsg(e.getMessage());
            pushRecordDTO.setReqSuccess(false);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            throw new Exception("聚水潭物料接口调用失败: " + e.getMessage(), e);
        }
    }

    private Map<String, Object> buildItemMap(UpdateGoodsRequest request) {
        UpdateGoodsRequest.Item item = request.getItems().get(0);
        Map<String, Object> itemMap = new HashMap<>();

        // 必填字段验证
        if (StringUtils.isEmpty(item.getSkuId()) || StringUtils.isEmpty(item.getIId()) || StringUtils.isEmpty(item.getName())) {
            throw new IllegalArgumentException(String.format(
                    "物料必填字段缺失 | skuId: %s, iId: %s, name: %s",
                    item.getSkuId(), item.getIId(), item.getName()
            ));
        }

        itemMap.put("sku_id", item.getSkuId());
        itemMap.put("i_id", item.getIId());
        itemMap.put("name", item.getName());
        return itemMap;
    }

    @Override
    @Transactional
    public JushuitanBaseResponse syncSupp(SupplierRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getSuppliers())) {
            String errorMsg = "syncSupp请求参数为空或suppliers列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        log.info("聚水潭供应商同步接口 - 请求开始 | 供应商数量: {}", request.getSuppliers().size());
        PushRecordDTO pushRecordDTO = buildPushRecordDTO(pushRecordInterType, docNo, docId);
        pushRecordDTO.setReqBody(request);

        try {
            // 2. 构建供应商列表
            List<Map<String, Object>> suppliers = request.getSuppliers().stream()
                    .map(this::buildSupplierItemMap)
                    .collect(Collectors.toList());

            String jsonRequest = JSON.toJSONString(suppliers);
            log.debug("聚水潭供应商同步接口 - 请求数据: {}", jsonRequest);
            pushRecordDTO.setReqBody(suppliers); // 保存结构化请求
            JushuitanBaseResponse response = jstService.getSellerItem(jsonRequest, "/open/supplier/upload", OutsidePushInter.INV_IDO_SYNC_TO_JST.name(), 11L, "22");
            log.debug("聚水潭供应商同步接口 - 响应: {}", response);

            // 4. 处理响应
            if (response == null) {
                log.error("聚水潭供应商同步接口 - 响应为空");
                pushRecordDTO.setReqSuccess(false);
                pushRecordDTO.setErrorMsg("接口返回为空");
                throw new RuntimeException("聚水潭接口返回为空");
            }


            pushRecordDTO.setReqSuccess(true);
            pushRecordDTO.setRespBody(response);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);

            log.info("聚水潭供应商同步接口 - 处理完成 | 供应商数量: {}, 状态: {}",
                    request.getSuppliers().size(), "成功");
            return response;

        } catch (Exception e) {
            log.error("聚水潭供应商同步接口异常 | 供应商数量: {}", request.getSuppliers().size(), e);
            pushRecordDTO.setErrorMsg(e.getMessage());
            pushRecordDTO.setReqSuccess(false);
            outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            throw new Exception("聚水潭供应商同步接口失败: " + e.getMessage(), e);
        }
    }

    @Transactional
    @Override
    public PurchaseInboundResponse purchaseinCreatebatch(PurchaseInboundListRequest request) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getList())) {
            String errorMsg = "purchaseinCreatebatch请求参数为空或list列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }
        log.info("聚水潭采购入库批量创建接口 - 请求开始 | 入库单数量: {}", request.getList().size());
        // PushRecordDTO pushRecordDTO = buildPushRecordDTO("JST", "purchaseinCreatebatch");
        // PushRecordDTO dto = new PushRecordDTO();
        // dto.setInterType(interType);
        // dto.setDocNo("");
        // dto.setDocId(0L);
        // dto.setReqSuccess(false);
        // pushRecordDTO.setReqBody(request);
        try {
            // 2. 构建请求数据 - 直接使用list，因为API期望的是数组格式
            String jsonRequest = JSON.toJSONString(request.getList());
            log.debug("聚水潭采购入库批量创建接口 - 请求数据: {}", jsonRequest);
            // pushRecordDTO.setReqBody(request.getList());

            // 3. 调用外部接口 - 使用正确的采购入库API端点
            JushuitanBaseResponse response = jstService.getSellerItem(jsonRequest, "/open/webapi/wmsapi/purchasein/createbatch", OutsidePushInter.INV_IDO_SYNC_TO_JST.name(), 11L, "22");
            log.debug("聚水潭采购入库批量创建接口 - 响应: {}", response);

            // 5. 解析响应为PurchaseInboundResponse
            PurchaseInboundResponse responseObj = (PurchaseInboundResponse) response;
            // if (responseObj == null) {
            //     log.error("聚水潭采购入库批量创建接口 - 响应解析失败");
            //     // pushRecordDTO.setReqSuccess(false);
            //     // pushRecordDTO.setErrorMsg("响应解析失败");
            //     throw new RuntimeException("响应解析失败");
            // }

            // // 6. 判断业务成功状态
            // boolean success = responseObj.isSuccess();
            // if (!success && responseObj.isTooFrequent()) {
            //     log.warn("聚水潭采购入库批量创建接口 - 调用频繁，错误码: {}, 消息: {}",
            //             responseObj.getCode(), responseObj.getMsg());
            // } else if (!success) {
            //     log.error("聚水潭采购入库批量创建接口 - 业务失败，错误码: {}, 消息: {}",
            //             responseObj.getCode(), responseObj.getMsg());
            // }
            //
            // // 7. 检查数据级别的成功状态
            // if (success && !CollectionUtils.isEmpty(responseObj.getData())) {
            //     List<String> failedExternalIds = responseObj.getData().stream()
            //             .filter(data -> !Boolean.TRUE.equals(data.getIs_success()))
            //             .map(PurchaseInboundResponse.PurchaseInboundData::getExternal_id)
            //             .collect(Collectors.toList());
            //
            //     if (!failedExternalIds.isEmpty()) {
            //         log.warn("聚水潭采购入库批量创建接口 - 部分入库单创建失败 | 失败的外部单号: {}",
            //                 String.join(",", failedExternalIds));
            //     }
            // }
            //
            // // pushRecordDTO.setReqSuccess(success);
            // // pushRecordDTO.setRespBody(response);
            // // outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            //
            // log.info("聚水潭采购入库批量创建接口 - 处理完成 | 入库单数量: {}, 状态: {}, 响应码: {}",
            //         request.getList().size(), success ? "成功" : "失败", responseObj.getCode());

            return responseObj;

        } catch (Exception e) {
            log.error("聚水潭采购入库批量创建接口异常 | 入库单数量: {}", request.getList().size(), e);
            // pushRecordDTO.setErrorMsg(e.getMessage());
            // pushRecordDTO.setReqSuccess(false);
            // outsidePushRecordRepository.savePushRecord(pushRecordDTO);
            throw new Exception("聚水潭采购入库批量创建接口失败: " + e.getMessage(), e);
        }
    }

    private PushRecordDTO buildPushRecordDTO(String interType, String docNo, Long docId) {
        PushRecordDTO dto = new PushRecordDTO();
        dto.setInterType(interType);
        dto.setDocNo("");
        dto.setDocId(0L);
        dto.setReqSuccess(false);
        return dto;
    }

    private Map<String, Object> buildSupplierItemMap(SupplierRequest.SupplierItem item) {
        Map<String, Object> map = new HashMap<>();
        map.put("supplier_code", item.getSupplier_code());
        map.put("name", item.getName());
        map.put("enabled", true);
        return map;
    }
}