package com.elitesland.zhiyuan.openapi.outer.nc.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.elitesland.zhiyuan.openapi.outer.nc.dto.NcResponseDTO;
import com.elitesland.zhiyuan.openapi.outer.nc.exception.NcApiException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

/**
 * NC响应处理工具类
 *
 * @author generated
 */
@Slf4j
public class NcResponseUtil {

    /**
     * 解析NC响应并检查结果
     *
     * @param responseJson NC接口返回的JSON字符串
     * @return 解析后的响应对象
     * @throws NcApiException 当NC接口返回失败状态时抛出
     */
    public static NcResponseDTO parseAndCheck(String responseJson) throws NcApiException {
        if (!StringUtils.hasText(responseJson)) {
            throw new NcApiException("NC接口返回空响应");
        }

        try {
            // 解析JSON响应
            NcResponseDTO response = JSON.parseObject(responseJson, NcResponseDTO.class);

            if (response == null) {
                log.error("NC接口响应解析失败: {}", responseJson);
                throw new NcApiException("NC接口响应解析失败");
            }

            log.info("NC接口响应解析结果: 状态={}, 消息={}, 单据类型={}, 动作={}",
                    response.getStatus(), response.getMessage(), response.getBilltype(), response.getAction());

            // // 不抛异常了
            // if (true) {
            //     return response;
            // }
            // 检查响应状态
            if (response.isFailed()) {
                String errorMessage = response.getErrorMessage();
                log.error("NC接口调用失败: {}", errorMessage);
                throw new NcApiException(response.getStatus(), response.getAction(),
                        response.getBilltype(), errorMessage);
            }

            if (!response.isSuccess()) {
                String errorMessage = String.format("NC接口返回未知状态: %s, 消息: %s",
                        response.getStatus(), response.getMessage());
                log.error(errorMessage);
                throw new NcApiException(errorMessage);
            }

            log.info("NC接口调用成功");
            return response;

        } catch (JSONException e) {
            log.error("NC接口响应JSON解析失败: {}", responseJson, e);
            throw new NcApiException("NC接口响应格式错误: " + e.getMessage(), e);
        }
    }

    /**
     * 解析NC响应（不抛出异常）
     *
     * @param responseJson NC接口返回的JSON字符串
     * @return 解析后的响应对象，解析失败时返回null
     */
    public static NcResponseDTO parseQuietly(String responseJson) {
        try {
            return parseAndCheck(responseJson);
        } catch (Exception e) {
            log.warn("NC响应解析失败: {}", e.getMessage());
            return null;
        }
    }

    /**
     * 检查响应是否成功
     *
     * @param responseJson NC接口返回的JSON字符串
     * @return 是否成功
     */
    public static boolean isSuccess(String responseJson) {
        NcResponseDTO response = parseQuietly(responseJson);
        return response != null && response.isSuccess();
    }

    /**
     * 获取错误信息
     *
     * @param responseJson NC接口返回的JSON字符串
     * @return 错误信息，成功时返回null
     */
    public static String getErrorMessage(String responseJson) {
        NcResponseDTO response = parseQuietly(responseJson);
        return response != null ? response.getErrorMessage() : "响应解析失败";
    }
}