package com.elitesland.zhiyuan.openapi.outer.jushuitan.service.impl;

import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.sys.outlog.OutsidePushInter;
import com.elitescloud.cloudt.sys.outlog.param.PushRecordDTO;
import com.elitescloud.cloudt.sys.outlog.rpc.OutsidePushRecordRepository;
import com.elitesland.zhiyuan.openapi.outer.jggyun.params.SupplierRequest;
import com.elitesland.zhiyuan.openapi.outer.jggyun.params.UpdateGoodsRequest;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.common.service.JstService;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.params.req.*;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.params.resp.*;
import com.elitesland.zhiyuan.openapi.outer.jushuitan.service.JushuitanService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
@RefreshScope
public class JushuitanServiceImpl implements JushuitanService {

    private final OutsidePushRecordRepository outsidePushRecordRepository;
    private final JstService jstService;

    @Override
    @Transactional
    public JushuitanBaseResponse createMaterial(UpdateGoodsRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        return createMaterial(null, null, null, request, pushRecordInterType, docId, docNo);
    }

    @Transactional
    @Override
    public JushuitanBaseResponse createMaterial(String appKey, String appSecret, String accessToken, UpdateGoodsRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getItems())) {
            String errorMsg = "createMaterial请求参数为空或items列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        if (request.getItems().size() > 1) {
            log.warn("createMaterial请求包含{}个商品，仅处理第一个商品。建议每次请求仅包含一个商品", request.getItems().size());
        }

        // 2. 关键日志：记录原始请求
        log.info("聚水潭物料接口 - 请求开始 | 商品ID: {}, 请求参数: {}",
                request.getItems().get(0).getSkuId(),
                JSON.toJSONString(request, true)); // 详细打印避免信息丢失
        try {
            // 3. 构建请求数据
            Map<String, Object> itemMap = buildItemMap(request);
            Map<String, Object> requestMap = Collections.singletonMap("items", Collections.singletonList(itemMap));
            String jsonRequest = JSON.toJSONString(requestMap);

            log.debug("聚水潭物料接口 - 请求数据: {}", jsonRequest);

            JushuitanBaseResponse response = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/jushuitan/itemsku/upload", OutsidePushInter.ITEM_SYNC_TO_JST.name(), docId, docNo);

            log.debug("聚水潭物料接口 - 响应: {}", response);

            // 5. 处理响应
            if (response == null) {
                log.error("聚水潭物料接口 - 响应为空");
                throw new RuntimeException("聚水潭接口返回为空");
            }
            log.info("聚水潭物料接口 - 处理完成 | 商品ID: {}, 响应状态: {}",
                    request.getItems().get(0).getSkuId(),
                    "成功");
            return response;

        } catch (Exception e) {
            log.error("聚水潭物料接口调用异常 | 商品ID: {}", request.getItems().get(0).getSkuId(), e);
            throw new Exception("聚水潭物料接口调用失败: " + e.getMessage(), e);
        }
    }

    private Map<String, Object> buildItemMap(UpdateGoodsRequest request) {
        UpdateGoodsRequest.Item item = request.getItems().get(0);
        Map<String, Object> itemMap = new HashMap<>();

        // 必填字段验证
        if (StringUtils.isEmpty(item.getSkuId()) || StringUtils.isEmpty(item.getIId()) || StringUtils.isEmpty(item.getName())) {
            throw new IllegalArgumentException(String.format(
                    "物料必填字段缺失 | skuId: %s, iId: %s, name: %s",
                    item.getSkuId(), item.getIId(), item.getName()
            ));
        }

        itemMap.put("sku_id", item.getSkuId());
        itemMap.put("i_id", item.getIId());
        itemMap.put("name", item.getName());
        return itemMap;
    }

    @Override
    @Transactional
    public SupplierSyncResponse syncSupp(SupplierRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        return syncSupp(null, null, null, request, pushRecordInterType, docId, docNo);
    }

    @Transactional
    @Override
    public SupplierSyncResponse syncSupp(String appKey, String appSecret, String accessToken, SupplierRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getSuppliers())) {
            String errorMsg = "syncSupp请求参数为空或suppliers列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        log.info("聚水潭供应商同步接口 - 请求开始 | 供应商数量: {}", request.getSuppliers().size());

        try {
            // 2. 构建供应商列表
            List<Map<String, Object>> suppliers = request.getSuppliers().stream()
                    .map(this::buildSupplierItemMap)
                    .collect(Collectors.toList());

            String jsonRequest = JSON.toJSONString(suppliers);
            log.debug("聚水潭供应商同步接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/supplier/upload", OutsidePushInter.SUPPORT_SYNC_TO_JST.name(), docId, docNo);
            log.debug("聚水潭供应商同步接口 - 响应: {}", baseResponse);

            // 4. 转换为SupplierSyncResponse
            SupplierSyncResponse response = convertToSupplierSyncResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                List<String> failedSuppliers = response.getData().getDatas().stream()
                        .filter(data -> !Boolean.TRUE.equals(data.getIssuccess()))
                        .map(SupplierSyncResponse.SupplierData::getSupplier_code)
                        .collect(Collectors.toList());

                if (!failedSuppliers.isEmpty()) {
                    log.warn("聚水潭供应商同步接口 - 部分供应商同步失败 | 失败的供应商编码: {}",
                            String.join(",", failedSuppliers));
                }
            }

            log.info("聚水潭供应商同步接口 - 处理完成 | 供应商数量: {}, 状态: {}, 响应码: {}",
                    request.getSuppliers().size(), response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭供应商同步接口异常 | 供应商数量: {}", request.getSuppliers().size(), e);
            throw new Exception("聚水潭供应商同步接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 转换基础响应为供应商同步响应
     */
    private SupplierSyncResponse convertToSupplierSyncResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        SupplierSyncResponse response = new SupplierSyncResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());
        // response.setCookie(baseResponse.getCookie());
        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为SupplierSyncData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                SupplierSyncResponse.SupplierSyncData supplierData = JSON.parseObject(dataJson, SupplierSyncResponse.SupplierSyncData.class);
                response.setData(supplierData);
            } catch (Exception e) {
                log.warn("转换供应商同步响应数据失败", e);
            }
        }
        return response;
    }

    @Transactional
    @Override
    public PurchaseInboundResponse purchaseinCreatebatch(PurchaseInboundListRequest request, Long docId, String docNo) throws Exception {
        return purchaseinCreatebatch(null, null, null, request, docId, docNo);
    }

    @Transactional
    @Override
    public PurchaseInboundResponse purchaseinCreatebatch(String appKey, String appSecret, String accessToken, PurchaseInboundListRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getList())) {
            String errorMsg = "purchaseinCreatebatch请求参数为空或list列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }
        log.info("聚水潭采购入库批量创建接口 - 请求开始 | 入库单数量: {}", request.getList().size());
        // try {
        // 2. 构建请求数据 - 直接使用list，因为API期望的是数组格式
        String jsonRequest = JSON.toJSONString(request.getList());
        log.debug("聚水潭采购入库批量创建接口 - 请求数据: {}", jsonRequest);
        // 3. 调用外部接口 - 使用正确的采购入库API端点
        JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/webapi/wmsapi/purchasein/createbatch", OutsidePushInter.INV_IDO_SYNC_TO_JST.name(), docId, docNo);
        log.debug("聚水潭采购入库批量创建接口 - 响应: {}", baseResponse);
        // 5. 解析响应为PurchaseInboundResponse
        // 4. 转换为PurchaseInboundResponse
        PurchaseInboundResponse response = JstService.convertToPurchaseInboundResponse(baseResponse);

        // 5. 业务状态检查
        if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData())) {
            List<String> failedMessage = response.getData().stream()
                    .filter(data -> !Boolean.TRUE.equals(data.getIs_success()))
                    .map(PurchaseInboundResponse.PurchaseInboundData::getMessage)
                    .collect(Collectors.toList());

            if (!failedMessage.isEmpty()) {
                log.warn("聚水潭采购入库批量创建接口 - 部分入库单创建失败 | 失败的外部单号: {}",
                        String.join(",", failedMessage));
            }
        }

        log.info("聚水潭采购入库批量创建接口 - 处理完成 | 入库单数量: {}, 状态: {}, 响应码: {}",
                request.getList().size(), response.isSuccess() ? "成功" : "失败", response.getCode());

        return response;
        // } catch (Exception e) {
        //     log.error("聚水潭采购入库批量创建接口异常 | 入库单数量: {}", request.getList().size(), e);
        //     throw new Exception("聚水潭采购入库批量创建接口失败: " + e.getMessage(), e);
        // }
    }

    private PushRecordDTO buildPushRecordDTO(String interType, String docNo, Long docId) {
        PushRecordDTO dto = new PushRecordDTO();
        dto.setInterType(interType);
        dto.setDocNo("");
        dto.setDocId(0L);
        dto.setReqSuccess(false);
        return dto;
    }

    @Override
    @Transactional
    public WarehouseQueryResponse queryWarehouse(WarehouseQueryRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        return queryWarehouse(null, null, null, request, pushRecordInterType, docId, docNo);
    }


    @Transactional
    @Override
    public WarehouseQueryResponse queryWarehouse(String appKey, String appSecret, String accessToken, WarehouseQueryRequest request, String pushRecordInterType, Long docId, String docNo) throws Exception {
        // 1. 参数验证（仓库查询的参数都是可选的）
        if (request == null) {
            request = new WarehouseQueryRequest();
        }

        log.info("聚水潭仓库查询接口 - 请求开始 | 页码: {}, 页大小: {}",
                request.getPage_index(), request.getPage_size());

        try {
            // 2. 构建请求参数
            Map<String, Object> requestMap = new HashMap<>();
            if (request.getPage_index() != null) {
                requestMap.put("page_index", request.getPage_index());
            }
            if (request.getPage_size() != null) {
                requestMap.put("page_size", request.getPage_size());
            }

            String jsonRequest = JSON.toJSONString(requestMap);
            log.debug("聚水潭仓库查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/wms/partner/query", pushRecordInterType, docId, docNo);
            log.debug("聚水潭仓库查询接口 - 响应: {}", baseResponse);

            // 4. 转换为WarehouseQueryResponse
            WarehouseQueryResponse response = convertToWarehouseQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭仓库查询接口 - 查询成功 | 仓库数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭仓库查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭仓库查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭仓库查询接口异常", e);
            throw new Exception("聚水潭仓库查询接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 转换基础响应为仓库查询响应
     */
    private WarehouseQueryResponse convertToWarehouseQueryResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        WarehouseQueryResponse response = new WarehouseQueryResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());
        // response.setCookie(baseResponse.getCookie());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为WarehouseQueryData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                WarehouseQueryResponse.WarehouseQueryData warehouseData = JSON.parseObject(dataJson, WarehouseQueryResponse.WarehouseQueryData.class);
                response.setData(warehouseData);
            } catch (Exception e) {
                log.warn("转换仓库查询响应数据失败", e);
            }
        }

        return response;
    }

    @Override
    @Transactional
    public PurchaseInboundQueryResponse queryPurchaseInbound(PurchaseInboundQueryRequest request, Long docId, String docNo) throws Exception {
        return queryPurchaseInbound(null, null, null, request, docId, docNo);
    }

    @Transactional
    @Override
    public PurchaseInboundQueryResponse queryPurchaseInbound(String appKey, String appSecret, String accessToken, PurchaseInboundQueryRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            request = new PurchaseInboundQueryRequest();
        }

        log.info("聚水潭采购入库查询接口 - 请求开始 | 参数: {}", JSON.toJSONString(request));

        try {
            // 2. 构建请求参数
            Map<String, Object> requestMap = buildPurchaseInboundQueryParams(request);

            String jsonRequest = JSON.toJSONString(requestMap);
            log.debug("聚水潭采购入库查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/webapi/wmsapi/purchasein/purchaseinquery", OutsidePushInter.JST_INV_IDO_RESULT_TO_SCM.name(), docId, docNo);
            log.debug("聚水潭采购入库查询接口 - 响应: {}", baseResponse);

            // 4. 转换为PurchaseInboundQueryResponse
            PurchaseInboundQueryResponse response = convertToPurchaseInboundQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭采购入库查询接口 - 查询成功 | 入库单数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭采购入库查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭采购入库查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭采购入库查询接口异常", e);
            throw new Exception("聚水潭采购入库查询接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 构建采购入库查询请求参数
     */
    private Map<String, Object> buildPurchaseInboundQueryParams(PurchaseInboundQueryRequest request) {
        Map<String, Object> requestMap = new HashMap<>();

        if (request.getPage_index() != null) {
            requestMap.put("page_index", request.getPage_index());
        }
        if (request.getPage_size() != null) {
            requestMap.put("page_size", request.getPage_size());
        }
        if (StringUtils.hasText(request.getModified_begin())) {
            requestMap.put("modified_begin", request.getModified_begin());
        }
        if (StringUtils.hasText(request.getModified_end())) {
            requestMap.put("modified_end", request.getModified_end());
        }
        if (!CollectionUtils.isEmpty(request.getPo_ids())) {
            requestMap.put("po_ids", request.getPo_ids());
        }
        if (!CollectionUtils.isEmpty(request.getIo_ids())) {
            requestMap.put("io_ids", request.getIo_ids());
        }
        if (!CollectionUtils.isEmpty(request.getStatuss())) {
            requestMap.put("statuss", request.getStatuss());
        }
        if (!CollectionUtils.isEmpty(request.getSo_ids())) {
            requestMap.put("so_ids", request.getSo_ids());
        }
        if (request.getStart_ts() != null) {
            requestMap.put("start_ts", request.getStart_ts());
        }
        if (request.getIs_get_total() != null) {
            requestMap.put("is_get_total", request.getIs_get_total());
        }
        if (request.getDate_type() != null) {
            requestMap.put("date_type", request.getDate_type());
        }
        if (!CollectionUtils.isEmpty(request.getSeller_ids())) {
            requestMap.put("seller_ids", request.getSeller_ids());
        }
        if (request.getOwner_co_id() != null) {
            requestMap.put("owner_co_id", request.getOwner_co_id());
        }

        return requestMap;
    }

    /**
     * 转换基础响应为采购入库查询响应
     */
    private PurchaseInboundQueryResponse convertToPurchaseInboundQueryResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        PurchaseInboundQueryResponse response = new PurchaseInboundQueryResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());
        // response.setCookie(baseResponse.getCookie());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为PurchaseInboundQueryData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                PurchaseInboundQueryResponse.PurchaseInboundQueryData queryData = JSON.parseObject(dataJson, PurchaseInboundQueryResponse.PurchaseInboundQueryData.class);
                response.setData(queryData);
            } catch (Exception e) {
                log.warn("转换采购入库查询响应数据失败", e);
            }
        }

        return response;
    }

    private Map<String, Object> buildSupplierItemMap(SupplierRequest.SupplierItem item) {
        Map<String, Object> map = new HashMap<>();
        map.put("supplier_code", item.getSupplier_code());
        map.put("name", item.getName());
        map.put("enabled", true);
        return map;
    }

    @Transactional
    @Override
    public PurchaseInboundResponse purchaseinReturnbatch(PurchaseReturnBatchRequest request, Long docId, String docNo) throws Exception {
        return purchaseinReturnbatch(null, null, null, request, docId, docNo);
    }

    @Transactional
    @Override
    public PurchaseInboundResponse purchaseinReturnbatch(String appKey, String appSecret, String accessToken, PurchaseReturnBatchRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getList())) {
            String errorMsg = "purchaseinCreatebatch请求参数为空或list列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }
        log.info("聚水潭采购退货批量创建接口 - 请求开始 | 入库单数量: {}", request.getList().size());
        // try {
        // 2. 构建请求数据 - 直接使用list，因为API期望的是数组格式
        String jsonRequest = JSON.toJSONString(request.getList());
        log.debug("聚水潭采购退批量创建接口 - 请求数据: {}", jsonRequest);
        // 3. 调用外部接口 - 使用正确的采购入库API端点
        JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/webapi/wmsapi/purchaseout/createbatch", OutsidePushInter.INV_ODO_SYNC_TO_JST.name(), docId, docNo);
        log.debug("聚水潭采购退批量创建接口 - 响应: {}", baseResponse);
        // 5. 解析响应为PurchaseInboundResponse
        // 4. 转换为PurchaseInboundResponse
        PurchaseInboundResponse response = JstService.convertToPurchaseInboundResponse(baseResponse);

        // 5. 业务状态检查
        if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData())) {
            List<String> failedMessage = response.getData().stream()
                    .filter(data -> !Boolean.TRUE.equals(data.getIs_success()))
                    .map(PurchaseInboundResponse.PurchaseInboundData::getMessage)
                    .collect(Collectors.toList());

            if (!failedMessage.isEmpty()) {
                log.warn("聚水潭采购退批量创建接口 - 部分入库单创建失败 | 失败的外部单号: {}",
                        String.join(",", failedMessage));
            }
        }

        log.info("聚水潭采购退批量创建接口 - 处理完成 | 出库单数量: {}, 状态: {}, 响应码: {}",
                request.getList().size(), response.isSuccess() ? "成功" : "失败", response.getCode());

        return response;
    }


    @Transactional
    @Override
    public PurchaseOnboundQueryResponse queryPurchaseOnbound(PurchaseInboundQueryRequest request, Long docId, String docNo) throws Exception {
        return queryPurchaseOnbound(null, null, null, request, docId, docNo);
    }

    @Transactional
    @Override
    public PurchaseOnboundQueryResponse queryPurchaseOnbound(String appKey, String appSecret, String accessToken, PurchaseInboundQueryRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            request = new PurchaseInboundQueryRequest();
        }

        log.info("聚水潭采购出库查询接口 - 请求开始 | 参数: {}", JSON.toJSONString(request));

        try {
            // 2. 构建请求参数
            Map<String, Object> requestMap = buildPurchaseOnboundQueryParams(request);

            String jsonRequest = JSON.toJSONString(requestMap);
            log.debug("聚水潭采购出库查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/purchaseout/query", OutsidePushInter.JST_INV_ODO_RESULT_TO_SCM.name(), docId, docNo);
            log.debug("聚水潭采购出库查询接口 - 响应: {}", baseResponse);

            // 4. 转换为PurchaseInboundQueryResponse
            PurchaseOnboundQueryResponse response = convertToPurchaseOnboundQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭采购出库查询接口 - 查询成功 | 入库单数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭采购出库查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭采购出库查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭采购出库查询接口失败", e);
            throw new Exception("聚水潭采购出库查询接口: " + e.getMessage(), e);
        }
    }


    /**
     * 转换基础响应为采购出库查询响应
     */
    private PurchaseOnboundQueryResponse convertToPurchaseOnboundQueryResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        PurchaseOnboundQueryResponse response = new PurchaseOnboundQueryResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());
        // response.setCookie(baseResponse.getCookie());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为PurchaseInboundQueryData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                PurchaseOnboundQueryResponse.PurchaseOnboundQueryData queryData = JSON.parseObject(dataJson, PurchaseOnboundQueryResponse.PurchaseOnboundQueryData.class);
                response.setData(queryData);
            } catch (Exception e) {
                log.warn("转换采购入库查询响应数据失败", e);
            }
        }

        return response;
    }


    /**
     * 构建采购出库查询请求参数
     */
    private Map<String, Object> buildPurchaseOnboundQueryParams(PurchaseInboundQueryRequest request) {
        Map<String, Object> requestMap = new HashMap<>();

        if (request.getPage_index() != null) {
            requestMap.put("page_index", request.getPage_index());
        }
        if (request.getPage_size() != null) {
            requestMap.put("page_size", request.getPage_size());
        }
        if (StringUtils.hasText(request.getModified_begin())) {
            requestMap.put("modified_begin", request.getModified_begin());
        }
        if (StringUtils.hasText(request.getModified_end())) {
            requestMap.put("modified_end", request.getModified_end());
        }
        if (!CollectionUtils.isEmpty(request.getPo_ids())) {
            requestMap.put("po_ids", request.getPo_ids());
        }
        if (!CollectionUtils.isEmpty(request.getIo_ids())) {
            requestMap.put("io_ids", request.getIo_ids());
        }
        if (!CollectionUtils.isEmpty(request.getStatuss())) {
            requestMap.put("statuss", request.getStatuss());
        }
        if (!CollectionUtils.isEmpty(request.getSo_ids())) {
            requestMap.put("so_ids", request.getSo_ids());
        }
        if (request.getStart_ts() != null) {
            requestMap.put("start_ts", request.getStart_ts());
        }
        if (request.getIs_get_total() != null) {
            requestMap.put("is_get_total", request.getIs_get_total());
        }
        if (request.getDate_type() != null) {
            requestMap.put("date_type", request.getDate_type());
        }
        if (!CollectionUtils.isEmpty(request.getSeller_ids())) {
            requestMap.put("seller_ids", request.getSeller_ids());
        }
        if (request.getOwner_co_id() != null) {
            requestMap.put("owner_co_id", request.getOwner_co_id());
        }

        return requestMap;
    }

    /**
     * 库存调拨上传（跨仓调拨）
     *
     * @param request 调拨上传请求参数
     * @param docId 文档ID
     * @param docNo 文档编号
     * @return {@link AllocateUploadResponse} 调拨上传响应
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public AllocateUploadResponse allocateUpload(AllocateUploadRequest request, Long docId, String docNo) throws Exception {
        return allocateUpload(null, null, null, request, docId, docNo);
    }

    /**
     * 库存调拨上传（跨仓调拨）
     *
     * @param request 调拨上传请求参数
     * @param docId 文档ID
     * @param docNo 文档编号
     * @return {@link AllocateUploadResponse} 调拨上传响应
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public AllocateUploadResponse allocateUpload(String appKey, String appSecret, String accessToken, AllocateUploadRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null || CollectionUtils.isEmpty(request.getList())) {
            String errorMsg = "allocateUpload请求参数为空或list列表为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        log.info("聚水潭库存调拨上传接口 - 请求开始 | 调拨单数量: {}", request.getList().size());

        try {
            // 2. 构建请求数据 - 直接使用list，因为API期望的是数组格式
            String jsonRequest = JSON.toJSONString(request.getList());
            log.debug("聚水潭库存调拨上传接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/allocate/kc/upload", OutsidePushInter.INV_ALLOCATE_SYNC_TO_JST.name(), docId, docNo);
            log.debug("聚水潭库存调拨上传接口 - 响应: {}", baseResponse);

            // 4. 转换为AllocateUploadResponse
            AllocateUploadResponse response = convertToAllocateUploadResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭库存调拨上传接口 - 上传成功 | 调拨单数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭库存调拨上传接口 - 上传成功但无数据");
            }

            log.info("聚水潭库存调拨上传接口 - 处理完成 | 调拨单数量: {}, 状态: {}, 响应码: {}",
                    request.getList().size(), response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭库存调拨上传接口异常 | 调拨单数量: {}", request.getList().size(), e);
            throw new Exception("聚水潭库存调拨上传接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 转换基础响应为库存调拨上传响应
     */
    private AllocateUploadResponse convertToAllocateUploadResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        AllocateUploadResponse response = new AllocateUploadResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为AllocateUploadData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                AllocateUploadResponse.AllocateUploadData allocateData = JSON.parseObject(dataJson, AllocateUploadResponse.AllocateUploadData.class);
                response.setData(allocateData);
            } catch (Exception e) {
                log.warn("转换库存调拨上传响应数据失败", e);
            }
        }

        return response;
    }

    /**
     * 调拨出查询
     * @param request
     * @param docId
     * @param docNo
     * @return
     * @throws Exception
     */
    @Transactional
    @Override
    public PurchAllotResultDetailQueryResponse queryAllotPurchaseOnbound(PurchaseInAllotQueryRequest request, Long docId, String docNo) throws Exception {
        return queryAllotPurchaseOnbound(null, null, null, request, docId, docNo);
    }

    @Transactional
    @Override
    public PurchAllotResultDetailQueryResponse queryAllotPurchaseOnbound(String appKey, String appSecret, String accessToken, PurchaseInAllotQueryRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            request = new PurchaseInAllotQueryRequest();
        }

        log.info("聚水潭调拨出库查询接口 - 请求开始 | 参数: {}", JSON.toJSONString(request));

        try {


            String jsonRequest = JSON.toJSONString(request);
            log.debug("聚水潭调拨出库查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/allocate/query", OutsidePushInter.JST_ALLOT_ODO_RESULT_TO_SCM.name(), docId, docNo);
            log.debug("聚水潭调拨出库查询接口 - 响应: {}", baseResponse);

            // 4. 转换为PurchaseInboundQueryResponse
            PurchAllotResultDetailQueryResponse response = convertToPurchAllotResultDetailQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭调拨出库查询接口 - 查询成功 | 入库单数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭调拨出库查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭调拨出库查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭采购出库查询接口失败", e);
            throw new Exception("聚水潭采购出库查询接口: " + e.getMessage(), e);
        }
    }
    /**
     * 转换基础响应为转换调拨出查询响应
     */
    private PurchAllotResultDetailQueryResponse convertToPurchAllotResultDetailQueryResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        PurchAllotResultDetailQueryResponse response = new PurchAllotResultDetailQueryResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为PurchaseInboundQueryData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                PurchAllotResultDetailQueryResponse.AllocateQueryData queryData = JSON.parseObject(dataJson, PurchAllotResultDetailQueryResponse.AllocateQueryData.class);
                response.setData(queryData);
            } catch (Exception e) {
                log.warn("转换调拨出查询响应数据失败", e);
            }
        }

        return response;
    }
    /**
     * 调拨入查询
     * @param request
     * @param docId
     * @param docNo
     * @return
     * @throws Exception
     */
    @Transactional
    @Override
    public PurchAllotResultDetailQueryResponse queryAllotPurchaseInbound(PurchaseInAllotQueryRequest request, Long docId, String docNo) throws Exception {
        return queryAllotPurchaseInbound(null, null, null, request, docId, docNo);
    }

    @Transactional
    @Override
    public PurchAllotResultDetailQueryResponse queryAllotPurchaseInbound(String appKey, String appSecret, String accessToken, PurchaseInAllotQueryRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            request = new PurchaseInAllotQueryRequest();
        }

        log.info("聚水潭调拨入库查询接口 - 请求开始 | 参数: {}", JSON.toJSONString(request));

        try {


            String jsonRequest = JSON.toJSONString(request);
            log.debug("聚水潭调拨入库查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/allocate/query", OutsidePushInter.JST_ALLOT_IN_RESULT_TO_SCM.name(), docId, docNo);
            log.debug("聚水潭调拨入库查询接口 - 响应: {}", baseResponse);

            // 4. 转换为PurchaseInboundQueryResponse
            PurchAllotResultDetailQueryResponse response = convertToPurchAllotResultDetailQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭调拨入库库查询接口 - 查询成功 | 入库单数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭调拨入库查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭调拨入库查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭调拨入库接口失败", e);
            throw new Exception("聚水潭调拨入库查询接口: " + e.getMessage(), e);
        }
    }

    /**
     * 其它出入库上传
     *
     * @param request 其它出入库上传请求参数
     * @param docId 文档ID
     * @param docNo 文档编号
     * @return {@link OtherInOutUploadResponse} 其它出入库上传响应
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public OtherInOutUploadResponse otherInOutUpload(OtherInOutUploadRequest request, Long docId, String docNo) throws Exception {
        return otherInOutUpload(null, null, null, request, docId, docNo);
    }

    /**
     * 其它出入库上传 (报废申请)
     *
     * @param appKey      应用密钥
     * @param appSecret   应用密钥
     * @param accessToken 访问令牌
     * @param request     请求
     * @param docId       文档Id
     * @param docNo       文件编号
     * @return {@link OtherInOutUploadResponse }
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public OtherInOutUploadResponse otherInOutUpload(String appKey, String appSecret, String accessToken, OtherInOutUploadRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            String errorMsg = "otherInOutUpload请求参数为空";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        if (StringUtils.isEmpty(request.getType()) || StringUtils.isEmpty(request.getExternal_id()) || CollectionUtils.isEmpty(request.getItems())) {
            String errorMsg = "otherInOutUpload必填参数缺失: type, external_id, items";
            log.error("参数校验失败: {}", errorMsg);
            throw new IllegalArgumentException(errorMsg);
        }

        log.info("聚水潭其它出入库上传接口 - 请求开始 | 类型: {}, 外部单号: {}, 商品数量: {}",
                request.getType(), request.getExternal_id(), request.getItems().size());

        try {
            // 2. 构建请求数据
            String jsonRequest = JSON.toJSONString(request);
            log.debug("聚水潭其它出入库上传接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/jushuitan/otherinout/upload", OutsidePushInter.OTHER_IN_OUT_SYNC_TO_JST.name(), docId, docNo);
            log.debug("聚水潭其它出入库上传接口 - 响应: {}", baseResponse);

            // 4. 转换为OtherInOutUploadResponse
            OtherInOutUploadResponse response = convertToOtherInOutUploadResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null) {
                log.info("聚水潭其它出入库上传接口 - 上传成功 | 外部单号: {}, 系统单号: {}",
                        response.getData().getExternal_id(), response.getData().getIo_id());
            } else if (response.isSuccess()) {
                log.info("聚水潭其它出入库上传接口 - 上传成功但无数据");
            }

            log.info("聚水潭其它出入库上传接口 - 处理完成 | 类型: {}, 外部单号: {}, 状态: {}, 响应码: {}",
                    request.getType(), request.getExternal_id(), response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭其它出入库上传接口异常 | 类型: {}, 外部单号: {}",
                    request.getType(), request.getExternal_id(), e);
            throw new Exception("聚水潭其它出入库上传接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 转换基础响应为其它出入库上传响应
     */
    private OtherInOutUploadResponse convertToOtherInOutUploadResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        OtherInOutUploadResponse response = new OtherInOutUploadResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为OtherInOutData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                OtherInOutUploadResponse.OtherInOutData otherInOutData = JSON.parseObject(dataJson, OtherInOutUploadResponse.OtherInOutData.class);
                response.setData(otherInOutData);
            } catch (Exception e) {
                log.warn("转换其它出入库上传响应数据失败", e);
            }
        }

        return response;
    }

    /**
     * 其它出入库查询
     *
     * @param request 其它出入库查询请求参数
     * @param docId 文档ID
     * @param docNo 文档编号
     * @return {@link OtherInOutQueryResponse} 其它出入库查询响应
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public OtherInOutQueryResponse queryOtherInOut(OtherInOutQueryRequest request, Long docId, String docNo) throws Exception {
        return queryOtherInOut(null, null, null, request, docId, docNo);
    }

    /**
     * 其它出入库查询
     *
     * @param appKey      应用密钥
     * @param appSecret   应用密钥
     * @param accessToken 访问令牌
     * @param request     请求
     * @param docId       文档Id
     * @param docNo       文件编号
     * @return {@link OtherInOutQueryResponse }
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public OtherInOutQueryResponse queryOtherInOut(String appKey, String appSecret, String accessToken, OtherInOutQueryRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            request = new OtherInOutQueryRequest();
        }

        log.info("聚水潭其它出入库查询接口 - 请求开始 | 参数: {}", JSON.toJSONString(request));

        try {
            // 2. 构建请求参数
            Map<String, Object> requestMap = buildOtherInOutQueryParams(request);

            String jsonRequest = JSON.toJSONString(requestMap);
            log.debug("聚水潭其它出入库查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/other/inout/query", OutsidePushInter.JST_FAIL_OUT_RESULT_TO_SCM.name(), docId, docNo);
            log.debug("聚水潭其它出入库查询接口 - 响应: {}", baseResponse);

            // 4. 转换为OtherInOutQueryResponse
            OtherInOutQueryResponse response = convertToOtherInOutQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭其它出入库查询接口 - 查询成功 | 单据数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭其它出入库查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭其它出入库查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭其它出入库查询接口异常", e);
            throw new Exception("聚水潭其它出入库查询接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 构建其它出入库查询请求参数
     */
    private Map<String, Object> buildOtherInOutQueryParams(OtherInOutQueryRequest request) {
        Map<String, Object> requestMap = new HashMap<>();

        if (StringUtils.hasText(request.getModified_begin())) {
            requestMap.put("modified_begin", request.getModified_begin());
        }
        if (StringUtils.hasText(request.getModified_end())) {
            requestMap.put("modified_end", request.getModified_end());
        }
        if (!CollectionUtils.isEmpty(request.getSo_ids())) {
            requestMap.put("so_ids", request.getSo_ids());
        }
        if (!CollectionUtils.isEmpty(request.getTypes())) {
            requestMap.put("types", request.getTypes());
        }
        if (StringUtils.hasText(request.getStatus())) {
            requestMap.put("status", request.getStatus());
        }
        if (request.getPage_index() != null) {
            requestMap.put("page_index", request.getPage_index());
        }
        if (request.getPage_size() != null) {
            requestMap.put("page_size", request.getPage_size());
        }
        if (request.getWms_co_id() != null) {
            requestMap.put("wms_co_id", request.getWms_co_id());
        }
        if (!CollectionUtils.isEmpty(request.getIo_ids())) {
            requestMap.put("io_ids", request.getIo_ids());
        }
        if (request.getDate_type() != null) {
            requestMap.put("date_type", request.getDate_type());
        }
        if (request.getStart_ts() != null) {
            requestMap.put("start_ts", request.getStart_ts());
        }
        if (request.getOwner_co_id() != null) {
            requestMap.put("owner_co_id", request.getOwner_co_id());
        }

        return requestMap;
    }

    /**
     * 转换基础响应为其它出入库查询响应
     */
    private OtherInOutQueryResponse convertToOtherInOutQueryResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        OtherInOutQueryResponse response = new OtherInOutQueryResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为OtherInOutQueryData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                OtherInOutQueryResponse.OtherInOutQueryData queryData = JSON.parseObject(dataJson, OtherInOutQueryResponse.OtherInOutQueryData.class);
                response.setData(queryData);
            } catch (Exception e) {
                log.warn("转换其它出入库查询响应数据失败", e);
            }
        }

        return response;
    }

    /**
     * 库存盘点查询
     *
     * @param request 库存盘点查询请求参数
     * @param docId 文档ID
     * @param docNo 文档编号
     * @return {@link InventoryCountQueryResponse} 库存盘点查询响应
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public InventoryCountQueryResponse queryInventoryCount(InventoryCountQueryRequest request, Long docId, String docNo) throws Exception {
        return queryInventoryCount(null, null, null, request, docId, docNo);
    }

    /**
     * 库存盘点查询
     *
     * @param appKey      应用密钥
     * @param appSecret   应用密钥
     * @param accessToken 访问令牌
     * @param request     请求
     * @param docId       文档Id
     * @param docNo       文件编号
     * @return {@link InventoryCountQueryResponse }
     * @throws Exception 例外
     */
    @Transactional
    @Override
    public InventoryCountQueryResponse queryInventoryCount(String appKey, String appSecret, String accessToken, InventoryCountQueryRequest request, Long docId, String docNo) throws Exception {
        // 1. 参数验证
        if (request == null) {
            request = new InventoryCountQueryRequest();
        }

        log.info("聚水潭库存盘点查询接口 - 请求开始 | 参数: {}", JSON.toJSONString(request));

        try {
            // 2. 构建请求参数
            Map<String, Object> requestMap = buildInventoryCountQueryParams(request);

            String jsonRequest = JSON.toJSONString(requestMap);
            log.debug("聚水潭库存盘点查询接口 - 请求数据: {}", jsonRequest);

            // 3. 调用外部接口并获取响应
            JushuitanBaseResponse baseResponse = jstService.getSellerItem(appKey, appSecret, accessToken, jsonRequest, "/open/inventory/count/query", OutsidePushInter.JST_INVENTORY_COUNT_RESULT_TO_SCM.name(), docId, docNo);
            log.debug("聚水潭库存盘点查询接口 - 响应: {}", baseResponse);

            // 4. 转换为InventoryCountQueryResponse
            InventoryCountQueryResponse response = convertToInventoryCountQueryResponse(baseResponse);

            // 5. 业务状态检查
            if (response.isSuccess() && response.getData() != null && !CollectionUtils.isEmpty(response.getData().getDatas())) {
                log.info("聚水潭库存盘点查询接口 - 查询成功 | 盘点单数量: {}", response.getData().getDatas().size());
            } else if (response.isSuccess()) {
                log.info("聚水潭库存盘点查询接口 - 查询成功但无数据");
            }

            log.info("聚水潭库存盘点查询接口 - 处理完成 | 状态: {}, 响应码: {}",
                    response.isSuccess() ? "成功" : "失败", response.getCode());

            return response;

        } catch (Exception e) {
            log.error("聚水潭库存盘点查询接口异常", e);
            throw new Exception("聚水潭库存盘点查询接口失败: " + e.getMessage(), e);
        }
    }

    /**
     * 构建库存盘点查询请求参数
     */
    private Map<String, Object> buildInventoryCountQueryParams(InventoryCountQueryRequest request) {
        Map<String, Object> requestMap = new HashMap<>();

        if (request.getPage_index() != null) {
            requestMap.put("page_index", request.getPage_index());
        }
        if (request.getPage_size() != null) {
            requestMap.put("page_size", request.getPage_size());
        }
        if (StringUtils.hasText(request.getModified_begin())) {
            requestMap.put("modified_begin", request.getModified_begin());
        }
        if (StringUtils.hasText(request.getModified_end())) {
            requestMap.put("modified_end", request.getModified_end());
        }
        if (StringUtils.hasText(request.getIo_ids())) {
            requestMap.put("io_ids", request.getIo_ids());
        }
        if (StringUtils.hasText(request.getStatus())) {
            requestMap.put("status", request.getStatus());
        }
        if (request.getOwner_co_id() != null) {
            requestMap.put("owner_co_id", request.getOwner_co_id());
        }

        return requestMap;
    }

    /**
     * 转换基础响应为库存盘点查询响应
     */
    private InventoryCountQueryResponse convertToInventoryCountQueryResponse(JushuitanBaseResponse baseResponse) {
        if (baseResponse == null) {
            return null;
        }

        InventoryCountQueryResponse response = new InventoryCountQueryResponse();
        response.setCode(baseResponse.getCode());
        response.setMsg(baseResponse.getMsg());
        response.setRequest_id(baseResponse.getRequest_id());
        response.setMsg_type(baseResponse.getMsg_type());
        response.setAct(baseResponse.getAct());

        // 转换data部分
        if (baseResponse.getData() != null) {
            try {
                // 将data转换为JSON字符串再解析为InventoryCountQueryData
                String dataJson = JSON.toJSONString(baseResponse.getData());
                InventoryCountQueryResponse.InventoryCountQueryData queryData = JSON.parseObject(dataJson, InventoryCountQueryResponse.InventoryCountQueryData.class);
                response.setData(queryData);
            } catch (Exception e) {
                log.warn("转换库存盘点查询响应数据失败", e);
            }
        }

        return response;
    }

}